import sys
import os
import time # Import time for sleep function
import pyperclip # For general text clipboard access
import win32clipboard # For specific Windows clipboard formats like CF_HDROP
import win32con # For clipboard constants
import requests # For making HTTP requests to the password generation API
import py7zr # For 7z compression and encryption

# Default encryption password
DEFAULT_PASSWORD = "password.r34.cc"
PASSWORD_GENERATION_API = "https://password.r34.cc/api/generate-password"

def get_clipboard_path_pywin32():
    """
    Attempts to get a file/folder path from the clipboard using pywin32 (CF_HDROP).
    Returns the path if successful, otherwise an empty string.
    """
    try:
        win32clipboard.OpenClipboard()
        if win32clipboard.IsClipboardFormatAvailable(win32con.CF_HDROP):
            hdrop = win32clipboard.GetClipboardData(win32con.CF_HDROP)
            if hdrop and len(hdrop) > 0:
                path = hdrop[0]
                return path
        return ""
    except Exception as e:
        return ""
    finally:
        try:
            win32clipboard.CloseClipboard()
        except Exception as e:
            pass

def get_default_folder_path():
    """
    Attempts to get a folder path from the clipboard.
    First tries pyperclip for general text, then pywin32 for CF_HDROP.
    If successful and the path is a valid directory, it returns the path.
    Otherwise, it returns None.
    """
    clipboard_content = ""
    
    # Try pyperclip first for general text content
    try:
        clipboard_content = pyperclip.paste()
        if clipboard_content:
            clipboard_content = clipboard_content.strip().strip('"')
            if os.path.isdir(clipboard_content):
                return clipboard_content
    except pyperclip.PyperclipException as e:
        pass
    except Exception as e:
        pass

    # If pyperclip didn't yield a valid directory, try pywin32 for CF_HDROP
    clipboard_content = get_clipboard_path_pywin32()
    
    if clipboard_content:
        if os.path.isdir(clipboard_content):
            return clipboard_content
    
    return None

def generate_new_password(master_password: str) -> str:
    """
    Generates a new random password using the password.r34.cc API.
    """
    try:
        headers = {
            'accept': 'application/json',
            'Content-Type': 'application/json',
        }
        data = {
            'master_password': master_password
        }
        response = requests.post(PASSWORD_GENERATION_API, headers=headers, json=data)
        response.raise_for_status() # Raise an exception for HTTP errors
        
        result = response.json()
        return result.get('password', '')
    except requests.exceptions.RequestException as e:
        print(f"Error generating new password from API: {e}", file=sys.stderr)
        return ""
    except Exception as e:
        print(f"An unexpected error occurred during password generation: {e}", file=sys.stderr)
        return ""

def compress_folder_to_7z(folder_path: str, output_archive_path: str, password: str):
    """
    Compresses a given folder into an encrypted 7z archive.
    """
    try:
        # Use filters for compression method and level.
        # For fastest compression, use LZMA2 with preset 1.
        # header_encryption=True encrypts filenames and other metadata.
        filters = [{'id': py7zr.FILTER_LZMA2, 'preset': 1}]
        with py7zr.SevenZipFile(output_archive_path, 'w', filters=filters, password=password, header_encryption=True) as archive:
            archive.writeall(folder_path, os.path.basename(folder_path))
        print(f"文件夹 '{folder_path}' 已成功以最快速度压缩并加密到 '{output_archive_path}'")
    except Exception as e:
        print(f"压缩文件夹时发生错误: {e}", file=sys.stderr)

def main():
    print("时效密码生成器 - 文件夹路径和密码设置")
    print("提示：您可以先复制一个文件夹到剪贴板，程序将尝试自动获取路径。")

    default_folder = get_default_folder_path()
    
    if default_folder:
        folder_path = input(f"请输入文件夹路径 (默认: {default_folder}): ")
        if not folder_path:
            folder_path = default_folder
    else:
        folder_path = input("请输入文件夹路径: ")

    if not os.path.isdir(folder_path):
        print(f"错误: 路径 '{folder_path}' 不是一个有效的文件夹。", file=sys.stderr)
        sys.exit(1)

    password = input(f"请输入加密密码 (默认: {DEFAULT_PASSWORD}): ")
    if not password:
        password = DEFAULT_PASSWORD

    print(f"\n您选择的文件夹路径是: {folder_path}")
    print(f"您设置的加密密码是: {password}")

    # Automatically set output archive name to folder name
    archive_name = f"{os.path.basename(folder_path)}.7z"
    output_archive_path = os.path.join(os.path.dirname(folder_path), archive_name)
    
    print(f"\n开始压缩文件夹 '{folder_path}' 到 '{output_archive_path}'...")
    compress_folder_to_7z(folder_path, output_archive_path, password)

    print("\n操作完成。")
    print(f"已将压缩文件路径 '{output_archive_path}' 复制到剪贴板，方便您粘贴。")
    pyperclip.copy(output_archive_path)

    print("\n可将下面文字发送给对方：")
    print("\n-----------------------------------------------------------")
    print("要解密此文件，请访问以下网站获取时效密码：")
    print("  https://password.r34.cc")
    print("在该网站输入您的主密码，即可获取用于解密文件的时效密码。")
    print("\n请务必注意：时效密码具有时间限制。")
    print("一旦超过有效时段，文件将永远无法解开。")
    print("如果发生这种情况，您将需要联系原文件主人重新提供相应文件。")
    print("\n-----------------------------------------------------------")
    input("按任意键退出...") # Wait for user input before exiting

if __name__ == "__main__":
    main()
